import { openConfigFile, saveConfigFile, getCurrentConfig, updateCurrentConfig } from './configManager.js';
import { generateH5PPackage } from './h5pGenerator.js';
import { initializeAIConceptionTab, copyPromptToClipboard, buildConfigFromAIForm, transformAiInputToEditorContent } from './aiAssistant.js';
import { renderEditor, createNewQuestionObject } from './editor.js';
import { initializePreview, destroyPreview, getLiveEditorChanges } from './preview.js';
import { downloadStandaloneHTML, copyStandaloneHTML } from './htmlGenerator.js';

function updateMemoryFromEditor() {
    const config = getCurrentConfig();
    if (!config || !config.editorContent) return;
    console.log("Mise à jour de la configuration depuis l'éditeur...");

    if (!config.editorContent.override) config.editorContent.override = {};
    config.editorContent.override.preventSkipping = document.getElementById('h5p-prevent-skipping').checked;
    config.editorContent.override.autoplay = document.getElementById('h5p-autoplay').checked;
    config.editorContent.override.loop = document.getElementById('h5p-loop').checked;

    const newInteractions = [];
    const questionCards = document.querySelectorAll('.question-card');

    questionCards.forEach(card => {
        const qIndex = parseInt(card.dataset.qIndex, 10);
        const originalInteraction = getCurrentConfig().editorContent.interactiveVideo.assets.interactions[qIndex];
        
        const correctRadio = card.querySelector(`input[name="correct-answer-${qIndex}"]:checked`);
        // L'index de la bonne réponse est la valeur du radio bouton sélectionné
        const correctIndexInUI = correctRadio ? parseInt(correctRadio.value, 10) : 0;
        
        // On récupère toutes les textareas de réponse
        const answerTextareas = Array.from(card.querySelectorAll('textarea.answer-input'));
        let answersText = answerTextareas.map(input => `<p>${input.value}</p>`);
        
        // On s'assure qu'il y a des réponses avant de manipuler le tableau
        if (answersText.length > 0 && correctIndexInUI < answersText.length) {
            // On retire la bonne réponse de sa position actuelle pour la mettre en premier
            const correctAnswerText = answersText.splice(correctIndexInUI, 1)[0];
            answersText.unshift(correctAnswerText);
        }
        
        const originalBehaviour = originalInteraction.action.params.behaviour ?? {};
        const updatedBehaviour = {
            ...originalBehaviour,
            autoContinue: card.querySelector('[data-type="behaviour-autocontinue"]').checked,
            enableRetry: card.querySelector('[data-type="behaviour-enableretry"]').checked,
            enableSolutionsButton: card.querySelector('[data-type="behaviour-enablesolutions"]').checked,
            soundEffectsEnabled: card.querySelector('[data-type="behaviour-soundeffects"]').checked,
            timeoutCorrect: parseInt(card.querySelector('[data-type="behaviour-timeoutcorrect"]').value, 10),
            timeoutWrong: parseInt(card.querySelector('[data-type="behaviour-timeoutwrong"]').value, 10),
        };
        
        const originalAdaptivity = originalInteraction.adaptivity ?? {};
        const updatedAdaptivity = {
            ...originalAdaptivity,
            requireCompletion: card.querySelector('[data-type="adaptivity-requirecompletion"]').checked,
            wrong: { seek: parseInt(card.querySelector('[data-type="seek-back-time"]').value, 10) }
        };

        const updatedInteraction = {
            ...originalInteraction,
            duration: { ...originalInteraction.duration, to: parseInt(card.querySelector('[data-type="appearance-time"]').value, 10) },
            action: {
                ...originalInteraction.action,
                params: {
                    ...originalInteraction.action.params,
                    choices: [{ question: `<p>${card.querySelector('[data-type="question-text"]').value}</p>`, answers: answersText }],
                    overallFeedback: [{ ...originalInteraction.action.params.overallFeedback[0], feedback: card.querySelector('[data-type="feedback"]').value }],
                    behaviour: updatedBehaviour
                }
            },
            adaptivity: updatedAdaptivity
        };
        newInteractions.push(updatedInteraction);
    });
    
    newInteractions.sort((a, b) => a.duration.to - b.duration.to);
    
    config.editorContent.interactiveVideo.assets.interactions = newInteractions;
    updateCurrentConfig(config);
}

document.addEventListener('DOMContentLoaded', () => {
    let activeTabHref = '#tab-ai';

    const burgerToggle = document.getElementById('burger-menu-toggle');
    const headerMenu = document.getElementById('header-actions-menu');

    burgerToggle.addEventListener('click', (event) => {
        event.stopPropagation();
        headerMenu.classList.toggle('open');
        burgerToggle.setAttribute('aria-expanded', headerMenu.classList.contains('open'));
    });

    document.addEventListener('click', (event) => {
        if (headerMenu.classList.contains('open') && !headerMenu.contains(event.target) && !burgerToggle.contains(event.target)) {
            headerMenu.classList.remove('open');
            burgerToggle.setAttribute('aria-expanded', 'false');
        }
    });

    const tabLinks = document.querySelectorAll('.tab-link');
    tabLinks.forEach(link => {
        link.addEventListener('click', (e) => {
            e.preventDefault();
            const newTabHref = link.getAttribute('href');
            if (newTabHref === activeTabHref) return;
            if (activeTabHref === '#tab-editor') updateMemoryFromEditor();
            if (activeTabHref === '#tab-preview') destroyPreview();
            activeTabHref = newTabHref;
            document.querySelector('.tab-link.active').classList.remove('active');
            link.classList.add('active');
            document.querySelector('.tab-pane.active').classList.remove('active');
            document.querySelector(activeTabHref).classList.add('active');
            if (activeTabHref === '#tab-preview') {
                updateMemoryFromEditor(); 
                renderEditor(getCurrentConfig()?.editorContent, updateMemoryFromEditor);
                if (window.YT && window.YT.Player) {
                    initializePreview(getCurrentConfig());
                } else {
                    window.onYouTubeIframeAPIReady = () => initializePreview(getCurrentConfig());
                }
            }
        });
    });

    initializeAIConceptionTab();
    renderEditor(null, updateMemoryFromEditor);

    document.getElementById('btn-open-config').addEventListener('click', () => {
        openConfigFile().then(config => {
            renderEditor(config?.editorContent, updateMemoryFromEditor);
        });
    });

    document.getElementById('btn-save-config').addEventListener('click', () => {
        if (activeTabHref === '#tab-editor') updateMemoryFromEditor();
        const configFromAI = buildConfigFromAIForm();
        const editorData = getCurrentConfig()?.editorContent;
        const finalConfig = { ...configFromAI, editorContent: editorData };
        updateCurrentConfig(finalConfig);
        saveConfigFile();
    });

    document.getElementById('btn-generate-h5p').addEventListener('click', () => {
        if (activeTabHref === '#tab-editor') updateMemoryFromEditor();
        const config = getCurrentConfig();
        generateH5PPackage(config);
    });

    document.getElementById('btn-download-html').addEventListener('click', () => {
        if (activeTabHref === '#tab-editor') updateMemoryFromEditor();
        const config = getCurrentConfig();
        downloadStandaloneHTML(config);
    });

    document.getElementById('btn-copy-html').addEventListener('click', () => {
        if (activeTabHref === '#tab-editor') updateMemoryFromEditor();
        const config = getCurrentConfig();
        copyStandaloneHTML(config);
    });

    document.getElementById('btn-copy-prompt').addEventListener('click', copyPromptToClipboard);
    
    document.getElementById('btn-load-ai-json').addEventListener('click', () => {
        const config = buildConfigFromAIForm();
        config.editorContent = transformAiInputToEditorContent(config);
        updateCurrentConfig(config);
        alert("Configuration chargée et transformée ! L'onglet Éditeur est prêt.");
        renderEditor(config.editorContent, updateMemoryFromEditor);
        document.querySelector('a[href="#tab-editor"]').click();
    });

    document.getElementById('btn-add-question').addEventListener('click', () => {
        const config = getCurrentConfig();
        if (!config || !config.editorContent) {
            alert("Veuillez d'abord charger une configuration avant d'ajouter une question.");
            return;
        }
        const newQuestion = createNewQuestionObject();
        config.editorContent.interactiveVideo.assets.interactions.push(newQuestion);
        config.editorContent.interactiveVideo.assets.interactions.sort((a, b) => a.duration.to - b.duration.to);
        updateCurrentConfig(config);
        renderEditor(config.editorContent, updateMemoryFromEditor);
    });

    const editorTab = document.getElementById('tab-editor');
    editorTab.addEventListener('click', (event) => {
        const targetButton = event.target.closest('button');
        if (!targetButton) return;

        const config = getCurrentConfig();
        if (!config || !config.editorContent) return;

        if (targetButton.classList.contains('delete-question-btn')) {
            const questionIndex = parseInt(targetButton.dataset.qIndex, 10);
            if (confirm(`Êtes-vous sûr de vouloir supprimer la question ${questionIndex + 1} ?`)) {
                config.editorContent.interactiveVideo.assets.interactions.splice(questionIndex, 1);
                updateCurrentConfig(config);
                renderEditor(config.editorContent, updateMemoryFromEditor);
            }
        }

        if (targetButton.classList.contains('delete-answer-btn')) {
            const qIndex = parseInt(targetButton.dataset.qIndex, 10);
            const aIndex = parseInt(targetButton.dataset.aIndex, 10);
            const interaction = config.editorContent.interactiveVideo.assets.interactions[qIndex];
            
            if (interaction.action.params.choices[0].answers.length <= 2) {
                alert("Une question doit avoir au moins deux choix de réponse.");
                return;
            }
            
            interaction.action.params.choices[0].answers.splice(aIndex, 1);
            updateCurrentConfig(config);
            renderEditor(config.editorContent, updateMemoryFromEditor);
        }
    });
    
    document.getElementById('btn-tester').addEventListener('click', () => {
        const changes = getLiveEditorChanges();
        if (!changes) {
            alert("Aucune question n'est en cours d'édition.");
            return;
        }
        const config = getCurrentConfig();
        if (!config) return;
        config.editorContent.interactiveVideo.assets.interactions[changes.index] = changes.updatedInteraction;
        config.editorContent.interactiveVideo.assets.interactions.sort((a, b) => a.duration.to - b.duration.to);
        updateCurrentConfig(config);
        renderEditor(config.editorContent, updateMemoryFromEditor);
        destroyPreview();
        initializePreview(config);
        alert("Changements appliqués et prévisualisation relancée !");
    });

    console.log("Application initialisée.");
});